# -*- coding: utf-8 -*-
#
# Copyright 2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# In addition, as a special exception, the copyright holders give
# permission to link the code of portions of this program with the
# OpenSSL library under certain conditions as described in each
# individual source file, and distribute linked combinations
# including the two.
# You must obey the GNU General Public License in all respects
# for all of the code used other than OpenSSL.  If you modify
# file(s) with this exception, you may extend this exception to your
# version of the file(s), but you are not obligated to do so.  If you
# do not wish to do so, delete this exception statement from your
# version.  If you delete this exception statement from all source
# files in the program, then also delete it here.
"""Tests for the Setup Account page Qt UI."""

from PyQt4 import QtGui, QtCore

from ubuntu_kylin_sso.qt import email_verification_page
from ubuntu_kylin_sso.qt.tests import (
    PageBaseTestCase,
    FakePageUiStyle,
)

# is ok to access private method/attrs in tests
# pylint: disable=W0212


class EmailVerificationTestCase(PageBaseTestCase):
    """Test the SetupAccountPage code."""

    ui_class = email_verification_page.EmailVerificationPage
    ui_signals = ('registrationSuccess',)
    ui_backend_signals = ('EmailValidated', 'EmailValidationError')

    def test_verification_code(self):
        """Test the verification value returned."""
        self.ui.ui.verification_code_edit.setText('asd123')
        value = self.ui.verification_code
        self.assertEqual(value, 'asd123')
        self.assertTrue(isinstance(value, str))

    def test_next_button(self):
        """Test the next button."""
        value = self.ui.next_button
        self.assertEqual(value, self.ui.ui.next_button)
        self.assertTrue(isinstance(value, QtGui.QPushButton))

    def test_connect_ui(self):
        """Test the connect ui method."""
        self.ui._connect_ui()
        # We expect 2 values because _connect_ui is called in the init too.
        self.assertEqual(self.ui.ui.verification_code_edit.receivers(
            QtCore.SIGNAL("textChanged(const QString&)")), 2)
        self.assertEqual(self.ui.next_button.receivers(
            QtCore.SIGNAL("clicked()")), 2)

    def test_validate_form_not_valid(self):
        """Test validate method."""
        button = FakePageUiStyle()
        self.patch(self.ui.ui, "next_button", button)
        self.ui.ui.verification_code_edit.setText('  ')
        self.ui.validate_form()
        self.assertFalse(button.isEnabled())

    def test_validate_form_valid(self):
        """Test validate method."""
        button = FakePageUiStyle()
        self.patch(self.ui.ui, "next_button", button)
        self.ui.ui.verification_code_edit.setText('asd123')
        self.ui.validate_form()
        self.assertTrue(button.isEnabled())

    def test_set_titles(self):
        """Test the set_titles method."""
        email = 'mail@example'
        self.ui.email = email
        self.ui.set_titles(email)
        self.assert_title_correct(email_verification_page.VERIFY_EMAIL_TITLE)
        expected = email_verification_page.VERIFY_EMAIL_CONTENT % \
            {"app_name": self.app_name,
             "email": email,
             }
        self.assert_subtitle_correct(expected)

    def test_initialize_page(self):
        """Test the initialization method."""
        button = FakePageUiStyle()
        self.patch(self.ui.ui, "next_button", button)
        self.ui.initializePage()
        self.assertTrue(button.properties['default'])
        self.assertFalse(button.isEnabled())
        expected = [('setButtonLayout', (([QtGui.QWizard.Stretch],), {}))]
        self.assertEqual(self.wizard.called, expected)

    def test_on_email_validation_error(self):
        """Test the validate_email method."""
        self.patch(self.ui, "show_error", self._set_called)
        error = {email_verification_page: 'error in email_verification_page'}

        self.ui.on_email_validation_error(self.app_name, error)

        expected = ((self.ui, ''), {})
        self.assertTrue(expected, self._called)
        self.assertEqual(self._overlay_hide_counter, 1)

    def test_on_email_validated(self):
        """Test the on_email_validated method."""
        email = 'email@example'
        self.ui.email = email

        self.assert_signal_emitted(self.ui.registrationSuccess, (email,),
            self.ui.on_email_validated, self.app_name, email)

    def test_validate_email(self):
        """Test the login method."""
        self.patch(self.ui, "hide_error", self._set_called)
        email = 'email@example.com'
        password = 'password'
        code = 'code'
        self.ui.email = email
        self.ui.password = password
        self.ui.ui.verification_code_edit.setText(code)

        self.ui.validate_email()

        self.assert_backend_called('validate_email',
            self.app_name, email, password, code)
        expected = ((), {})
        self.assertEqual(expected, self._called)
        self.assertEqual(self._overlay_show_counter, 1)
