/************************************************************************/
/*                                                                      */
/*    vspline - a set of generic tools for creation and evaluation      */
/*              of uniform b-splines                                    */
/*                                                                      */
/*            Copyright 2015, 2016 by Kay F. Jahnke                     */
/*                                                                      */
/*    Permission is hereby granted, free of charge, to any person       */
/*    obtaining a copy of this software and associated documentation    */
/*    files (the "Software"), to deal in the Software without           */
/*    restriction, including without limitation the rights to use,      */
/*    copy, modify, merge, publish, distribute, sublicense, and/or      */
/*    sell copies of the Software, and to permit persons to whom the    */
/*    Software is furnished to do so, subject to the following          */
/*    conditions:                                                       */
/*                                                                      */
/*    The above copyright notice and this permission notice shall be    */
/*    included in all copies or substantial portions of the             */
/*    Software.                                                         */
/*                                                                      */
/*    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND    */
/*    EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES   */
/*    OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND          */
/*    NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT       */
/*    HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,      */
/*    WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING      */
/*    FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR     */
/*    OTHER DEALINGS IN THE SOFTWARE.                                   */
/*                                                                      */
/************************************************************************/

/// \file grok.cc
///
/// \brief demonstrates use of vspline::grok_type

#include <iostream>
#include <vspline/vspline.h>

// sin_eval implements an eval-type function typical for vspline

void sin_eval ( const double & input , double & output )
{
  output = sin ( input ) ;
}

// t_sin_eval is an equivalent template, it can evaluate SIMD data
// just as non-vectorized types

template < typename T >
void t_sin_eval ( const T & input , T & output )
{
  output = sin ( input ) ;
}

struct sin_uf
: public vspline::unary_functor < double , double >
{
  template < typename IN , typename OUT >
  void eval ( const IN & in , OUT & out ) const
  {
    out = sin ( in ) ;
  }
} ;

int main ( int argc , char * argv[] )
{
  typedef vspline::grok_type < double , double > grok_t ;
  
  // we create a first grok_type object passing sin_eval to the
  // constructor. sin_eval has the functor syntax typical for vspline:
  // it takes both it's arguments per reference an has void return type.
  // Note how, with Vc in use, this grok_type object will broadcast
  // sin_eval to process vectorized types
  
  grok_t gk1 ( sin_eval ) ;

  // grok_type is quite flexible and can also take 'ordinary' functions.
  // again, with the single-argument constructor, sin will be broadcast. 
  
  grok_t gk2 ( sin ) ;

  // and it can take vspline::unary_functors as constructor argument as well.
  // since sin_uf uses a template to implement it's eval routine, it actually
  // uses vector code when evaluating vector data. Hence the grok_type
  // created from it will also use vector code.

  sin_uf suf ;  
  grok_t gk3 ( suf ) ;  

  double x = 1.0 ;
  
  std::cout << x << " -> " << gk1 ( x ) << std::endl ;
  std::cout << x << " -> " << gk2 ( x ) << std::endl ;
  std::cout << x << " -> " << gk3 ( x ) << std::endl ;

  // argtype will be an SIMD type if Vc is used, otherwise it will
  // simply be double.

  typedef typename grok_t::in_v argtype ;
  
  argtype xx = 1.0 ;
  
  std::cout << xx << " -> " << gk1 ( xx ) << std::endl ;
  std::cout << xx << " -> " << gk2 ( xx ) << std::endl ;
  std::cout << xx << " -> " << gk3 ( xx ) << std::endl ;
  
#ifdef USE_VC
  
  // here we use a two-argument constructor for a grok_type.
  // we pass a lambda expression for the vectorized evaluation,
  // and the evaluation will use vector code.
  
  grok_t gk4 ( sin , [] ( const argtype & x ) { return sin(x) ; } ) ;
  std::cout << xx << " -> " << gk4 ( xx ) << std::endl ;
  
  // another example for using the two-argument constructor
  // here we specialize t_sin_eval to provide acceptable aguments
  
  grok_t gk5 ( t_sin_eval<double> , t_sin_eval<argtype> ) ;
  std::cout << xx << " -> " << gk5 ( xx ) << std::endl ;
  
#endif
}
