/* 
 * Copyright © 2012-2013 Collabora, Ltd.
 * 
 * Permission to use, copy, modify, distribute, and sell this
 * software and its documentation for any purpose is hereby granted
 * without fee, provided that the above copyright notice appear in
 * all copies and that both that copyright notice and this permission
 * notice appear in supporting documentation, and that the name of
 * the copyright holders not be used in advertising or publicity
 * pertaining to distribution of the software without specific,
 * written prior permission.  The copyright holders make no
 * representations about the suitability of this software for any
 * purpose.  It is provided "as is" without express or implied
 * warranty.
 * 
 * THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS
 * SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS, IN NO EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF
 * THIS SOFTWARE.
 */

#ifndef SUBSURFACE_SERVER_PROTOCOL_H
#define SUBSURFACE_SERVER_PROTOCOL_H

#ifdef  __cplusplus
extern "C" {
#endif

#include <stdint.h>
#include <stddef.h>
#include "wayland-util.h"

struct wl_client;
struct wl_resource;

struct wl_subcompositor;
struct wl_subsurface;

extern const struct wl_interface wl_subcompositor_interface;
extern const struct wl_interface wl_subsurface_interface;

#ifndef WL_SUBCOMPOSITOR_ERROR_ENUM
#define WL_SUBCOMPOSITOR_ERROR_ENUM
enum wl_subcompositor_error {
	WL_SUBCOMPOSITOR_ERROR_BAD_SURFACE = 0,
};
#endif /* WL_SUBCOMPOSITOR_ERROR_ENUM */

/**
 * wl_subcompositor - sub-surface compositing
 * @destroy: unbind from the subcompositor interface
 * @get_subsurface: give a surface the role sub-surface
 *
 * The global interface exposing sub-surface compositing capabilities. A
 * wl_surface, that has sub-surfaces associated, is called the parent
 * surface. Sub-surfaces can be arbitrarily nested and create a tree of
 * sub-surfaces.
 *
 * The root surface in a tree of sub-surfaces is the main surface. The main
 * surface cannot be a sub-surface, because sub-surfaces must always have a
 * parent.
 *
 * A main surface with its sub-surfaces forms a (compound) window. For
 * window management purposes, this set of wl_surface objects is to be
 * considered as a single window, and it should also behave as such.
 *
 * The aim of sub-surfaces is to offload some of the compositing work
 * within a window from clients to the compositor. A prime example is a
 * video player with decorations and video in separate wl_surface objects.
 * This should allow the compositor to pass YUV video buffer processing to
 * dedicated overlay hardware when possible.
 */
struct wl_subcompositor_interface {
	/**
	 * destroy - unbind from the subcompositor interface
	 *
	 * Informs the server that the client will not be using this
	 * protocol object anymore. This does not affect any other objects,
	 * wl_subsurface objects included.
	 */
	void (*destroy)(struct wl_client *client,
			struct wl_resource *resource);
	/**
	 * get_subsurface - give a surface the role sub-surface
	 * @id: the new subsurface object id
	 * @surface: the surface to be turned into a sub-surface
	 * @parent: the parent surface
	 *
	 * Create a sub-surface interface for the given surface, and
	 * associate it with the given parent surface. This turns a plain
	 * wl_surface into a sub-surface.
	 *
	 * The to-be sub-surface must not already have a dedicated purpose,
	 * like any shell surface type, cursor image, drag icon, or
	 * sub-surface. Otherwise a protocol error is raised.
	 */
	void (*get_subsurface)(struct wl_client *client,
			       struct wl_resource *resource,
			       uint32_t id,
			       struct wl_resource *surface,
			       struct wl_resource *parent);
};

#ifndef WL_SUBSURFACE_ERROR_ENUM
#define WL_SUBSURFACE_ERROR_ENUM
enum wl_subsurface_error {
	WL_SUBSURFACE_ERROR_BAD_SURFACE = 0,
};
#endif /* WL_SUBSURFACE_ERROR_ENUM */

/**
 * wl_subsurface - sub-surface interface to a wl_surface
 * @destroy: remove sub-surface interface
 * @set_position: reposition the sub-surface
 * @place_above: restack the sub-surface
 * @place_below: restack the sub-surface
 * @set_sync: set sub-surface to synchronized mode
 * @set_desync: set sub-surface to desynchronized mode
 *
 * An additional interface to a wl_surface object, which has been made a
 * sub-surface. A sub-surface has one parent surface. A sub-surface's size
 * and position are not limited to that of the parent. Particularly, a
 * sub-surface is not automatically clipped to its parent's area.
 *
 * A sub-surface becomes mapped, when a non-NULL wl_buffer is applied and
 * the parent surface is mapped. The order of which one happens first is
 * irrelevant. A sub-surface is hidden if the parent becomes hidden, or if
 * a NULL wl_buffer is applied. These rules apply recursively through the
 * tree of surfaces.
 *
 * The behaviour of wl_surface.commit request on a sub-surface depends on
 * the sub-surface's mode. The possible modes are synchronized and
 * desynchronized, see methods wl_subsurface.set_sync and
 * wl_subsurface.set_desync. Synchronized mode caches the wl_surface state
 * to be applied when the parent's state gets applied, and desynchronized
 * mode applies the pending wl_surface state directly. A sub-surface is
 * initially in the synchronized mode.
 *
 * Sub-surfaces have also other kind of state, which is managed by
 * wl_subsurface requests, as opposed to wl_surface requests. This state
 * includes the sub-surface position relative to the parent surface
 * (wl_subsurface.set_position), and the stacking order of the parent and
 * its sub-surfaces (wl_subsurface.place_above and .place_below). This
 * state is applied when the parent surface's wl_surface state is applied,
 * regardless of the sub-surface's mode. As the exception, set_sync and
 * set_desync are effective immediately.
 *
 * The main surface can be thought to be always in desynchronized mode,
 * since it does not have a parent in the sub-surfaces sense.
 *
 * Even if a sub-surface is in desynchronized mode, it will behave as in
 * synchronized mode, if its parent surface behaves as in synchronized
 * mode. This rule is applied recursively throughout the tree of surfaces.
 * This means, that one can set a sub-surface into synchronized mode, and
 * then assume that all its child and grand-child sub-surfaces are
 * synchronized, too, without explicitly setting them.
 *
 * If the wl_surface associated with the wl_subsurface is destroyed, the
 * wl_subsurface object becomes inert. Note, that destroying either object
 * takes effect immediately. If you need to synchronize the removal of a
 * sub-surface to the parent surface update, unmap the sub-surface first by
 * attaching a NULL wl_buffer, update parent, and then destroy the
 * sub-surface.
 *
 * If the parent wl_surface object is destroyed, the sub-surface is
 * unmapped.
 */
struct wl_subsurface_interface {
	/**
	 * destroy - remove sub-surface interface
	 *
	 * The sub-surface interface is removed from the wl_surface
	 * object that was turned into a sub-surface with
	 * wl_subcompositor.get_subsurface request. The wl_surface's
	 * association to the parent is deleted, and the wl_surface loses
	 * its role as a sub-surface. The wl_surface is unmapped.
	 */
	void (*destroy)(struct wl_client *client,
			struct wl_resource *resource);
	/**
	 * set_position - reposition the sub-surface
	 * @x: coordinate in the parent surface
	 * @y: coordinate in the parent surface
	 *
	 * This schedules a sub-surface position change. The sub-surface
	 * will be moved so, that its origin (top-left corner pixel) will
	 * be at the location x, y of the parent surface coordinate system.
	 * The coordinates are not restricted to the parent surface area.
	 * Negative values are allowed.
	 *
	 * The next wl_surface.commit on the parent surface will reset the
	 * sub-surface's position to the scheduled coordinates.
	 *
	 * The initial position is 0, 0.
	 */
	void (*set_position)(struct wl_client *client,
			     struct wl_resource *resource,
			     int32_t x,
			     int32_t y);
	/**
	 * place_above - restack the sub-surface
	 * @sibling: the reference surface
	 *
	 * This sub-surface is taken from the stack, and put back just
	 * above the reference surface, changing the z-order of the
	 * sub-surfaces. The reference surface must be one of the sibling
	 * surfaces, or the parent surface. Using any other surface,
	 * including this sub-surface, will cause a protocol error.
	 *
	 * The z-order is double-buffered state, and will be applied on the
	 * next commit of the parent surface. See wl_surface.commit and
	 * wl_subcompositor.get_subsurface.
	 *
	 * A new sub-surface is initially added as the top-most in the
	 * stack of its siblings and parent.
	 */
	void (*place_above)(struct wl_client *client,
			    struct wl_resource *resource,
			    struct wl_resource *sibling);
	/**
	 * place_below - restack the sub-surface
	 * @sibling: the reference surface
	 *
	 * The sub-surface is placed just below of the reference surface.
	 * See wl_subsurface.place_above.
	 */
	void (*place_below)(struct wl_client *client,
			    struct wl_resource *resource,
			    struct wl_resource *sibling);
	/**
	 * set_sync - set sub-surface to synchronized mode
	 *
	 * Change the commit behaviour of the sub-surface to synchronized
	 * mode, also described as the parent dependant mode.
	 *
	 * In synchronized mode, wl_surface.commit on a sub-surface will
	 * accumulate the committed state in a cache, but the state will
	 * not be applied and hence will not change the compositor output.
	 * The cached state is applied to the sub-surface immediately after
	 * the parent surface's state is applied. This ensures atomic
	 * updates of the parent and all its synchronized sub-surfaces.
	 * Applying the cached state will invalidate the cache, so further
	 * parent surface commits do not (re-)apply old state.
	 *
	 * See wl_subsurface for the recursive effect of this mode.
	 */
	void (*set_sync)(struct wl_client *client,
			 struct wl_resource *resource);
	/**
	 * set_desync - set sub-surface to desynchronized mode
	 *
	 * Change the commit behaviour of the sub-surface to
	 * desynchronized mode, also described as independent or freely
	 * running mode.
	 *
	 * In desynchronized mode, wl_surface.commit on a sub-surface will
	 * apply the pending state directly, without caching, as happens
	 * normally with a wl_surface. Calling wl_surface.commit on the
	 * parent surface has no effect on the sub-surface's wl_surface
	 * state. This mode allows a sub-surface to be updated on its own.
	 *
	 * If cached state exists when wl_surface.commit is called in
	 * desynchronized mode, the pending state is added to the cached
	 * state, and applied as whole. This invalidates the cache.
	 *
	 * Note: even if a sub-surface is set to desynchronized, a parent
	 * sub-surface may override it to behave as synchronized. For
	 * details, see wl_subsurface.
	 *
	 * If a surface's parent surface behaves as desynchronized, then
	 * the cached state is applied on set_desync.
	 */
	void (*set_desync)(struct wl_client *client,
			   struct wl_resource *resource);
};

#ifdef  __cplusplus
}
#endif

#endif
